(function($, window, document, undefined) {
    "use strict";
    if (!window.requestAnimationFrame) {
        window.requestAnimationFrame = function() {
            return window.webkitRequestAnimationFrame || window.mozRequestAnimationFrame || window.oRequestAnimationFrame || window.msRequestAnimationFrame || function(callback, element) {
                window.setTimeout(callback, 1e3 / 60)
            }
        }()
    }
    var qrr, QRCodeReader = function() {};
    $.qrCodeReader = {
        jsQRpath: "js/jsQR.min.js",
        beepPath: "js/beep.mp3",
        instance: null,
        defaults: {
            multiple: false,
            qrcodeRegexp: /./,
            audioFeedback: true,
            repeatTimeout: 1500,
            target: null,
            skipDuplicates: true,
            lineColor: "#FF3B58",
            callback: function(code) {}
        }
    };
    QRCodeReader.prototype = {
        constructor: QRCodeReader,
        init: function() {
            qrr.buildHTML();
            qrr.scriptLoaded = false;
            qrr.isOpen = false;
            $.getScript($.qrCodeReader.jsQRpath, function(data, textStatus, jqxhr) {
                if (jqxhr.status == 200) {
                    qrr.scriptLoaded = true
                } else {
                    console.error("Error loading QRCode parser script")
                }
            })
        },
        buildHTML: function() {
            qrr.bgOverlay = $('<div id="qrr-overlay"></div>');
            qrr.container = $('<div id="qrr-container"></div>');
            qrr.closeBtn = $('<span id="qrr-close">&times;</span>');
            qrr.closeBtn.appendTo(qrr.container);
            qrr.okBtn = $('<a id="qrr-ok">OK</a>');
            qrr.loadingMessage = $('<div id="qrr-loading-message">🎥 Unable to access video stream (please make sure you have a webcam enabled)</div>');
            qrr.canvas = $('<canvas id="qrr-canvas" class="hidden"></canvas>');
            qrr.audio = $('<audio hidden id="qrr-beep" src="' + $.qrCodeReader.beepPath + '" type="audio/mp3"></audio>');
            qrr.outputDiv = $('<div id="qrr-output"></div>');
            qrr.outputNoData = $('<div id="qrr-nodata">No QR code detected.</div>');
            qrr.outputData = $('<div id="qrr-output-data"></div>');
            qrr.outputNoData.appendTo(qrr.outputDiv);
            qrr.outputData.appendTo(qrr.outputDiv);
            qrr.loadingMessage.appendTo(qrr.container);
            qrr.canvas.appendTo(qrr.container);
            qrr.outputDiv.appendTo(qrr.container);
            qrr.audio.appendTo(qrr.container);
            qrr.okBtn.appendTo(qrr.container);
            qrr.bgOverlay.appendTo(document.body);
            qrr.bgOverlay.on("click", qrr.close);
            qrr.closeBtn.on("click", qrr.close);
            qrr.container.appendTo(document.body);
            qrr.video = document.createElement("video")
        },
        drawLine: function(begin, end, color) {
            var canvas = qrr.canvas[0].getContext("2d");
            canvas.beginPath();
            canvas.moveTo(begin.x, begin.y);
            canvas.lineTo(end.x, end.y);
            canvas.lineWidth = 4;
            canvas.strokeStyle = color;
            canvas.stroke()
        },
        drawBox: function(location, color) {
            qrr.drawLine(location.topLeftCorner, location.topRightCorner, color);
            qrr.drawLine(location.topRightCorner, location.bottomRightCorner, color);
            qrr.drawLine(location.bottomRightCorner, location.bottomLeftCorner, color);
            qrr.drawLine(location.bottomLeftCorner, location.topLeftCorner, color)
        },
        setOptions: function(element, options) {
            var dataOptions = {
                multiple: $(element).data("qrr-multiple"),
                qrcodeRegexp: new RegExp($(element).data("qrr-qrcode-regexp")),
                audioFeedback: $(element).data("qrr-audio-feedback"),
                repeatTimeout: $(element).data("qrr-repeat-timeout"),
                target: $(element).data("qrr-target"),
                skipDuplicates: $(element).data("qrr-skip-duplicates"),
                lineColor: $(element).data("qrr-line-color"),
                callback: $(element).data("qrr-callback")
            };
            options = $.extend({}, dataOptions, options);
            var settings = $.extend({}, $.qrCodeReader.defaults, options);
            $(element).data("qrr", settings)
        },
        getOptions: function(element) {
            qrr.settings = $(element).data("qrr")
        },
        open: function() {
            if (qrr.isOpen) return;
            qrr.getOptions(this);
            qrr.bgOverlay.show();
            qrr.container.slideDown();
            qrr.codes = [];
            qrr.outputNoData.show();
            qrr.outputData.empty();
            qrr.outputData.hide();
            if (qrr.settings.multiple) {
                qrr.okBtn.show();
                qrr.okBtn.off("click").on("click", qrr.doneReading)
            } else {
                qrr.okBtn.hide()
            }
            $(document).on("keyup.qrCodeReader", function(e) {
                if (e.keyCode === 27) {
                    qrr.close()
                }
                if (qrr.settings.multiple && e.keyCode === 13) {
                    qrr.doneReading()
                }
            });
            qrr.isOpen = true;
            if (qrr.scriptLoaded) {
                qrr.start()
            }
        },
        start: function() {
            navigator.mediaDevices.getUserMedia({
                video: {
                    facingMode: "environment"
                }
            }).then(function(stream) {
                qrr.video.srcObject = stream;
                qrr.video.setAttribute("playsinline", true);
                qrr.video.play();
                qrr.startReading()
            })
        },
        startReading: function() {
            qrr.requestID = window.requestAnimationFrame(qrr.read)
        },
        doneReading: function() {
            var value = qrr.codes[0];
            if (qrr.settings.target) {
                if (qrr.settings.multiple) {
                    var value = qrr.codes.join("\n")
                }
                $(qrr.settings.target).val(value)
            }
            if (qrr.settings.callback) {
                try {
                    if (qrr.settings.multiple) {
                        qrr.settings.callback(qrr.codes)
                    } else {
                        qrr.settings.callback(value)
                    }
                } catch (err) {
                    console.error(err)
                }
            }
			if($(".lx-pagination ul").attr("data-table") !== "ramassages" && $(".lx-pagination ul").attr("data-table") === "colisretour"){
				qrr.close();
			}
			else{
				window.setTimeout(function(){
					navigator.mediaDevices.getUserMedia({
						video: {
							facingMode: "environment"
						}
					}).then(function(stream) {
						qrr.video.srcObject = stream;
						qrr.video.setAttribute("playsinline", true);
						qrr.video.play();
						qrr.startReading();
					});				
				},2000);				
			}
        },
        read: function() {
            var codeRead = false;
            var canvas = qrr.canvas[0].getContext("2d");
            qrr.loadingMessage.text("⌛ Loading video...");
            qrr.canvas.off("click.qrCodeReader", qrr.startReading);
            if (qrr.video.readyState === qrr.video.HAVE_ENOUGH_DATA) {
                qrr.loadingMessage.hide();
                qrr.canvas.removeClass("hidden");
                qrr.canvas[0].height = qrr.video.videoHeight;
                qrr.canvas[0].width = qrr.video.videoWidth;
                canvas.drawImage(qrr.video, 0, 0, qrr.canvas[0].width, qrr.canvas[0].height);
                var imageData = canvas.getImageData(0, 0, qrr.canvas[0].width, qrr.canvas[0].height);
                var code = jsQR(imageData.data, imageData.width, imageData.height, {
                    inversionAttempts: "dontInvert"
                });
                if (code && qrr.settings.qrcodeRegexp.test(code.data)) {
                    qrr.drawBox(code.location, qrr.settings.lineColor);
                    codeRead = true;
                    qrr.codes.push(code.data);
                    qrr.outputNoData.hide();
                    qrr.outputData.show();
                    if (qrr.settings.audioFeedback) {
                        qrr.audio[0].play()
                    }
                    if (qrr.settings.multiple) {
                        if (qrr.settings.skipDuplicates) {
                            qrr.codes = $.unique(qrr.codes)
                        }
                        $('<div class="qrr-input"></div>').text(code.data).appendTo(qrr.outputData);
                        qrr.outputDiv[0].scrollTop = qrr.outputDiv[0].scrollHeight;
                        qrr.canvas.on("click.qrCodeReader", qrr.startReading);
                        if (qrr.settings.repeatTimeout > 0) {
                            setTimeout(qrr.startReading, qrr.settings.repeatTimeout)
                        } else {
                            qrr.loadingMessage.text("Click on the image to read the next QRCode");
                            qrr.loadingMessage.show()
                        }
                    } else {
                        qrr.doneReading()
                    }
                }
            }
            if (!codeRead) {
                qrr.startReading()
            }
        },
        close: function() {
            if (qrr.requestID) {
                window.cancelAnimationFrame(qrr.requestID)
            }
            $(document).off("keyup.qrCodeReader");
            if (qrr.video.srcObject) {
                qrr.video.srcObject.getTracks()[0].stop()
            }
            qrr.canvas.addClass("hidden");
            qrr.loadingMessage.show();
            qrr.bgOverlay.hide();
            qrr.container.hide();
            qrr.isOpen = false
        }
    };
    $.fn.qrCodeReader = function(options) {
        if (!$.qrCodeReader.instance) {
            qrr = new QRCodeReader;
            qrr.init();
            $.qrCodeReader.instance = qrr
        }
        return this.each(function() {
            qrr.setOptions(this, options);
            $(this).off("click.qrCodeReader").on("click.qrCodeReader", qrr.open)
        })
    }
})(jQuery, window, document);